function [z]= zernfun_full(m,n,r,theta)
% z= uses cos and sin to compute zernike polynomials 
% z1= uses absolute values to compute zernike polynomials 

%ZERNFUN Zernike functions of order N and frequency M on the unit circle.
%   Z = ZERNFUN(M,N,R,THETA) returns the Zernike functions of order M
%   and angular frequency N, evaluated at positions (R,THETA) on the
%   unit circle.  M is a vector of positive integers (including 0), and
%   N is a vector with the same number of elements as M.  Each element
%   k of N must be a positive integer, with possible values N(k) = -M(k)
%   to +M(k) in steps of 2.  R is a vector of numbers between 0 and 1,
%   and THETA is a vector of angles.  R and THETA must have the same
%   length.  The output Z is a matrix with one column for every (M,N)
%   pair, and one row for every (R,THETA) pair.
%
%   Z = ZERNFUN(M,N,R,THETA,'norm') returns the normalized Zernike
%   functions.  The normalization factor sqrt((2-delta(n,0))*(m+1)/pi),
%   with delta(n,0) the Kronecker delta, is chosen so that the integral
%   of (r * [Zmn(r,theta)]^2) over the unit circle (from r=0 to r=1,
%   and theta=0 to theta=2*pi) is unity.  For the non-normalized
%   polynomials, max(Zmn(r=1,theta))=1 for all [m,n].
%
%   The Zernike functions are an orthogonal basis on the unit circle.
%   They are used in disciplines such as astronomy, optics, and
%   optometry to describe functions on a circular domain.
%
%   The following table lists the first 15 Zernike functions.
%
%       m    n    Zernike function             Normalization
%       ----------------------------------------------------
%       0    0    1                              1/sqrt(pi)
%       1    1    r * cos(theta)                 2/sqrt(pi)
%       1   -1    r * sin(theta)                 2/sqrt(pi)
%       2    2    r^2 * cos(2*theta)             sqrt(6/pi)
%       2    0    (2*r^2 - 1)                    sqrt(3/pi)
%       2   -2    r^2 * sin(2*theta)             sqrt(6/pi)
%       3    3    r^3 * cos(3*theta)             sqrt(8/pi)
%       3    1    (3*r^3 - 2*r) * cos(theta)     sqrt(8/pi)
%       3   -1    (3*r^3 - 2*r) * sin(theta)     sqrt(8/pi)
%       3   -3    r^3 * sin(3*theta)             sqrt(8/pi)
%       4    4    r^4 * cos(4*theta)             sqrt(10/pi)
%       4    2    (4*r^4 - 3*r^2) * cos(2*theta) sqrt(10/pi)
%       4    0    6*r^4 - 6*r^2 + 1              sqrt(5/pi)
%       4   -2    (4*r^4 - 3*r^2) * sin(2*theta) sqrt(10/pi)
%       4   -4    r^4 * sin(4*theta)             sqrt(10/pi)
%       ----------------------------------------------------
%
%   Example 1:
%
%       % Display the Zernike function Z(m=5,n=1)
%       x = -1:0.01:1;
%       [X,Y] = meshgrid(x,x);
%       [theta,r] = cart2pol(X,Y);
%       idx = r<=1;
%       z = nan(size(X));
%       z(idx) = zernfun(5,1,r(idx),theta(idx));
%       figure
%       pcolor(x,x,z), shading interp
%       axis square, colorbar
%       title('Zernike function Z_5^1(r,\theta)')
%
%   Example 2:
%
%       % Display the first 10 Zernike functions
%       x = -1:0.01:1;
%       [X,Y] = meshgrid(x,x);
%       [theta,r] = cart2pol(X,Y);
%       idx = r<=1;
%       z = nan(size(X));
%       m = [0  1  1  2  2  2  3  3  3  3];
%       n = [0 -1  1 -2  0  2 -3 -1  1  3];
%       Nplot = [4 10 12 16 18 20 22 24 26 28];
%       y = zernfun(m,n,r(idx),theta(idx));
%       figure('Units','normalized')
%       for k = 1:10
%           z(idx) = y(:,k);
%           subplot(4,7,Nplot(k))
%           pcolor(x,x,z), shading interp
%           set(gca,'XTick',[],'YTick',[])
%           axis square
%           title(['Z_{' num2str(m(k)) '}^{' num2str(n(k)) '}'])
%       end
%
%   See also ZERNPOL, ZERNFUN2.
%   Paul Fricker 2/28/2012
% Check and prepare the inputs:
% -----------------------------
if ( ~any(size(m)==1) ) || ( ~any(size(n)==1) )
    error('zernfun:NMvectors','N and M must be vectors.')
end
if length(m)~=length(n)
    error('zernfun:NMlength','N and M must be the same length.')
end
m = m(:);
n = n(:);
if any(mod(m-n,2))
    error('zernfun:NMmultiplesof2', ...
          'All N and M must differ by multiples of 2 (including 0).')
end
if any(n>m)
    error('zernfun:MlessthanN', ...
          'Each M must be less than or equal to its corresponding N.')
end
% if any( r>1 | r<0 )
%     error('zernfun:Rlessthan1','All R must be between 0 and 1.')
% end
if ( ~any(size(r)==1) ) || ( ~any(size(theta)==1) )
    error('zernfun:RTHvector','R and THETA must be vectors.')
end
r = r(:);
theta = theta(:);
length_r = length(r);
if length_r~=length(theta)
    error('zernfun:RTHlength', ...
          'The number of R- and THETA-values must be equal.')
end
% Check normalization:
% --------------------
if nargin==5 && ischar(nflag)
    isnorm = strcmpi(nflag,'norm');
    if ~isnorm
        error('zernfun:normalization','Unrecognized normalization flag.')
    end
else
    isnorm = false;
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute the Zernike Polynomials
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Determine the required powers of r:
% -----------------------------------
n_abs = abs(n);
rpowers = [];
for j = 1:length(m)
    rpowers = [rpowers n_abs(j):2:m(j)];
end
rpowers = unique(rpowers);
% Pre-compute the values of r raised to the required powers,
% and compile them in a matrix:
% -----------------------------
if rpowers(1)==0
    rpowern = arrayfun(@(p)r.^p,rpowers(2:end),'UniformOutput',false);
    rpowern = cat(2,rpowern{:});
    rpowern = [ones(length_r,1) rpowern];
else
    rpowern = arrayfun(@(p)r.^p,rpowers,'UniformOutput',false);
    rpowern = cat(2,rpowern{:});
end
% Compute the values of the polynomials:
% --------------------------------------
z = zeros(length_r,length(m));
for j = 1:length(m)
    s = 0:(m(j)-n_abs(j))/2;
    pows = m(j):-2:n_abs(j);
    for k = length(s):-1:1
        p = (1-2*mod(s(k),2))* ...
                   prod(2:(m(j)-s(k)))/              ...
                   prod(2:s(k))/                     ...
                   prod(2:((m(j)-n_abs(j))/2-s(k)))/ ...
                   prod(2:((m(j)+n_abs(j))/2-s(k)));
        idx = (pows(k)==rpowers);
        z(:,j) = z(:,j) + p*rpowern(:,idx);
    end
    
    if isnorm
        z(:,j) = z(:,j)*sqrt((1+(n(j)~=0))*(m(j)+1)/pi);
    end
end
z1=z;

% END: Compute the Zernike Polynomials
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute the Zernike functions:
% ------------------------------
%% Case 1 :
idx_pos = n>0;
idx_neg = n<0;
if any(idx_pos)
    z(:,idx_pos) = z(:,idx_pos).*cos(theta*n_abs(idx_pos)');
end  
if any(idx_neg)
    z(:,idx_neg) = z(:,idx_neg).*sin(theta*n_abs(idx_neg)');
end






   
end
%EOF zernfun
